import sys, numpy as np
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileProgram, compileShader

# ---------- Shaders ----------
VERTEX_SRC = """
#version 330
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){
    texCoord = (pos + 1.0)*0.5;
    gl_Position = vec4(pos,0,1);
}
"""

FRAGMENT_SRC = """
#version 330
in vec2 texCoord;
out vec4 fragColor;

// Uniforms
uniform float cycle;
uniform float omegaTime;
uniform float phi;
uniform float phiInv;
uniform float fibTable[128];
uniform float primeTable[128];

const int INSTANCE_COUNT = 64;

float prismatic_instance(int id, float r){
    float phi_harm = pow(phi, mod(id,16)) / 10.0;       // normalized
    float fib_harm = fibTable[id%128] / 100.0;          // normalized
    float dyadic = float(1 << (id%16)) / 1024.0;        // normalized
    float prime_harm = primeTable[id%128] / 256.0;      // normalized
    float Omega = 0.5 + 0.5*sin(omegaTime + float(id)*0.05);
    float r_dim = pow(r, float((id%7)+1) / 4.0);        // softer exponent
    return sqrt(phi_harm * fib_harm * dyadic * prime_harm * Omega) * r_dim;
}

void main(){
    float r = length(texCoord - 0.5) * 2.0;

    // Superimpose 64 instances
    float val = 0.0;
    for(int i=0;i<INSTANCE_COUNT;i++){
        val += prismatic_instance(i,r);
    }
    val /= float(INSTANCE_COUNT); // normalize final value

    float phase = sin(cycle*0.01 + val*6.28); // phase mapping

    // Color mapping: prismatic analog
    fragColor = vec4(val, phase*0.5 + 0.5, 1.0 - val, 1.0);
}
"""

# ---------- Globals ----------
window = None
shader = None
vao = None
cycle = 0.0
omega_time = 0.0

# Precompute lookup tables
phi = 1.6180339887
fib_table = np.array([((phi**n - (1-phi)**n)/np.sqrt(5)) for n in range(128)], dtype=np.float32)
prime_table = np.array([2,3,5,7,11,13,17,19,23,29,31,37,41,43,47,53,59,61,67,71,73,79,83,89,97,
                        101,103,107,109,113,127,131,137,139,149,151,157,163,167,173,179,181,191,193,197,199], dtype=np.float32)

# ---------- OpenGL Init ----------
def init_gl():
    global shader, vao
    shader = compileProgram(compileShader(VERTEX_SRC, GL_VERTEX_SHADER),
                            compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER))
    verts = np.array([-1,-1,1,-1,-1,1,1,-1,1,1,-1,1], dtype=np.float32)
    vao = glGenVertexArrays(1)
    glBindVertexArray(vao)
    vbo = glGenBuffers(1)
    glBindBuffer(GL_ARRAY_BUFFER,vbo)
    glBufferData(GL_ARRAY_BUFFER, verts.nbytes, verts, GL_STATIC_DRAW)
    glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0,None)
    glEnableVertexAttribArray(0)

    glUseProgram(shader)
    glUniform1f(glGetUniformLocation(shader,"phi"), phi)
    glUniform1f(glGetUniformLocation(shader,"phiInv"), 1/phi)
    glUniform1fv(glGetUniformLocation(shader,"fibTable"),128,fib_table)
    glUniform1fv(glGetUniformLocation(shader,"primeTable"),128,prime_table)

# ---------- Display ----------
def display():
    global cycle, omega_time
    glClear(GL_COLOR_BUFFER_BIT)
    glUseProgram(shader)
    glUniform1f(glGetUniformLocation(shader,"cycle"), cycle)
    glUniform1f(glGetUniformLocation(shader,"omegaTime"), omega_time)
    glBindVertexArray(vao)
    glDrawArrays(GL_TRIANGLES,0,6)
    glutSwapBuffers()
    cycle += 1.0
    omega_time += 0.02

# ---------- Idle ----------
def idle():
    glutPostRedisplay()

# ---------- Main ----------
def main():
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL 64-Instance Prismatic Fabric")
    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(idle)
    glutMainLoop()

if __name__=="__main__":
    main()
